import { NextRequest, NextResponse } from 'next/server';
import { getComprehensiveDatabase, initComprehensiveDatabase } from '@/src/lib/comprehensiveDatabase';
import { DomainFormSchema } from '@/src/utils/comprehensiveValidators';

export async function GET(request: NextRequest, { params }: { params: { id: string } }) {
    try {
        await initComprehensiveDatabase();
        const db = getComprehensiveDatabase();

        const { id } = await params;
        const domain = db.prepare(`
            SELECT d.*, c.name as customer_name, c.email as customer_email
            FROM domains d
            LEFT JOIN customers c ON d.customer_id = c.id
            WHERE d.id = ?
        `).get(id);

        if (!domain) {
            return NextResponse.json({ error: 'Domain not found' }, { status: 404 });
        }

        return NextResponse.json(domain);
    } catch (error) {
        console.error('Error fetching domain:', error);
        return NextResponse.json({ error: 'Failed to fetch domain' }, { status: 500 });
    }
}

export async function PUT(request: NextRequest, { params }: { params: { id: string } }) {
    try {
        await initComprehensiveDatabase();
        const db = getComprehensiveDatabase();

        const { id } = await params;
        const body = await request.json();

        // Validate the request body
        const validatedData = DomainFormSchema.parse(body);

        const now = new Date().toISOString();

        const result = db.prepare(`
            UPDATE domains 
            SET customer_id = ?, domain_name = ?, website = ?, owner = ?, email = ?, package = ?, 
                amount = ?, status = ?, priority = ?, next_payment = ?, notes = ?, updated_at = ?
            WHERE id = ?
        `).run(
            validatedData.customer_id,
            validatedData.domain_name,
            validatedData.website || null,
            validatedData.owner || null,
            validatedData.email || null,
            validatedData.package || null,
            validatedData.amount || null,
            validatedData.status,
            validatedData.priority,
            validatedData.next_payment || null,
            validatedData.notes || null,
            now,
            id
        );

        if (result.changes === 0) {
            return NextResponse.json({ error: 'Domain not found' }, { status: 404 });
        }

        const updatedDomain = db.prepare(`
            SELECT d.*, c.name as customer_name, c.email as customer_email
            FROM domains d
            LEFT JOIN customers c ON d.customer_id = c.id
            WHERE d.id = ?
        `).get(id);

        return NextResponse.json(updatedDomain);
    } catch (error) {
        console.error('Error updating domain:', error);
        if (error instanceof Error && error.name === 'ZodError') {
            return NextResponse.json({ error: 'Validation error', details: error.message }, { status: 400 });
        }
        return NextResponse.json({ error: 'Failed to update domain' }, { status: 500 });
    }
}

export async function DELETE(request: NextRequest, { params }: { params: { id: string } }) {
    try {
        await initComprehensiveDatabase();
        const db = getComprehensiveDatabase();

        const { id } = await params;
        const result = db.prepare('DELETE FROM domains WHERE id = ?').run(id);

        if (result.changes === 0) {
            return NextResponse.json({ error: 'Domain not found' }, { status: 404 });
        }

        return NextResponse.json({ message: 'Domain deleted successfully' });
    } catch (error) {
        console.error('Error deleting domain:', error);
        return NextResponse.json({ error: 'Failed to delete domain' }, { status: 500 });
    }
}
